function [density,zdensity,ndata,gyro,vol,xb,yb] = dhist3(data,bins,mod,out)
%This function takes a 2D array and creates a 2D density histogram.
%Synatax:   [density,zdensity,ndata,gyro,vol,xb,yb] = dhist3(data,bins,mod,out);
%Input:     bins = the number of bins in the histogram.
%           data = the array to be binned
%           mod = the modifier.  This is a conversion factor for changing
%               the units of the histogram, e.g., 1 pixel = 100nm, thus
%               mod=100.  Default = 1.
%           out = remove outlier, 5 SD from median.
%Output:    density = a vector of densities equal to bins.
%           zprofile = the z score of each bin
%           rcount = raw count in each bin
%           vol = a vector of volumes equal to bins.
%           xb = the x scale for the 2d hist
%           yb = the y scale for the 2d hist
%           gyro = a structure that contains the raw, volume, density and
%                   plot of the gyro visualization.

if nargin<3     %defalut mod
    mod = 1;
    out = 0;
end

data = data.*mod;       %change the units if needed.

data = unique(data,'rows');     %make sure the data is cleaned of NaNs

[ndata,c] = hist3(data(:,[1 2]),bins);    %create the binned data array

xb = linspace(min(data(:,1)),max(data(:,1)),size(ndata,1));       %x scale for the 2d hist
yb = linspace(min(data(:,2)),max(data(:,2)),size(ndata,2));       %y scale for the 2d hist

figure('Name','Raw Count Profile','NumberTitle','off');h = imagesc(xb,yb,ndata');       %plot this graph first.

%now figure out x, y & z to calcuate the volumn cube, rational here is that
%in x, y space the space is the same, only in z is there a volume
%difference.
x = repmat(c{1,1},bins(1),1);      %map the x to the spatial fields
y = repmat(c{1,2}',1,bins(2));     %map the y to the spatial fields
x = vectorize(x);               %vectorize x
y = vectorize(y);               %vectorize y
origin = zeros(size(x,1),2);    %create the origin array, 0,0
z_len = dddist([x y],origin);   %calculate the z length for the array
x_len = abs(c{1,1}(3)-c{1,1}(1));   %calculate the x length for the array
y_len = abs(c{1,2}(3)-c{1,1}(1));   %calculate the y length for the array
%now convert the length to field arrays
z_len = reshape(z_len,bins(1),bins(2))';
x_len = repmat(x_len,bins(1),bins(2));
y_len = repmat(y_len,bins(1),bins(2));
%now calculate the volume
vol = z_len.*x_len.*y_len;
%now calculate the density
density = ndata./vol;       %get the density value.

%now calculate the gyro view: new volume combine the sides of the rotations
%then mirror.
r = floor(bins(2)/2)-1;       %calculate the rows to flip: -1 for indexing purposes
r2 = ceil(bins(2)/2);      %forward indexing
right = fliplr(ndata(:,end-r:end));   %get to right half of the data and flip it
rvol = fliplr(vol(:,end-r:end));    %do the same for the volume
if rem(bins(2),2)       %if the divison is not even pad one row
    right = horzcat(right,zeros(1,bins(1)));
    rvol = horzcat(rvol,zeros(1,bins(1)));
end
gdata = ndata(:,1:r2)+right;   %gyro data
gvol = vol(:,1:r2)+rvol;        %gyro volume
gdensity = gdata./gvol;     %gyro density
gplot = horzcat(gdensity,fliplr(gdensity(:,1:r+1)));  %mirror the graph for plotting
%gprob = gplot./(sum(sum(gplot)));   %calculate the probability plot
%output the gyro data
gyro.raw = gdata;
gyro.vol = gvol;
gyro.density = gdensity;
gyro.plot = gplot;
%gyro.prob = gprob;

%outlier
if out
    out_thresh = vectorize(density);
    out_thresh(out_thresh==0) = [];
    out_thresh = std(out_thresh)*10;
    density(density>out_thresh) = out_thresh;
    %process gplot too
    out_thresh = vectorize(gplot);
    out_thresh(out_thresh==0) = [];
    out_thresh = std(out_thresh)*10;
    gplot(gplot>out_thresh) = out_thresh;
end

figure('Name','Density Profile','NumberTitle','off');h2 = imagesc(xb,yb,density');    %graph the density
%figure('Name','Probability Density Profile','NumberTitle','off');h2 = imagesc(xb,yb,gprob');    %graph the probability
figure('Name','Raw Folded Profile','NumberTitle','off');h2 = imagesc(xb,yb,gdata');    %graph the density
figure('Name','Folded Volume','NumberTitle','off');h2 = imagesc(xb,yb,gvol');    %graph the density
figure('Name','Folded Density Profile','NumberTitle','off');h2 = imagesc(xb,yb,gdensity');    %graph the density
figure('Name','Mirror Density Profile','NumberTitle','off');h2 = imagesc(xb,yb,gplot');    %graph the density
sdensity = csaps({xb,yb},gplot',[],{xb,yb});     %cubic spline smooth the images
figure('Name','Cubic Spline Smoothed Density Profile','NumberTitle','off');h3 = imagesc(xb,yb,sdensity);   %graph
%figure('Name','Volume Profile','NumberTitle','off');h4 = imagesc(xb,yb,vol');       %plot volume

%calculate statics: z score
%first calculate the mean and the standard deviation of the density distribution
m = mean2(gplot);
s = std2(gplot);
%now calculate the z score for each bin
zdensity = (gplot-m)./s;  %here the z score
%now plot it
figure('Name','Z Score Profile','NumberTitle','off');h5 = imagesc(xb,yb,zdensity');    %graph the density